using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.AddressableAssets;
using Unity.Industrial.Forma.Core;
using Unity.Industrial.Forma.Core.AddressableSources;

#if UNITY_EDITOR
using UnityEditor.AddressableAssets;
#endif

/// <summary>
/// Used to match and evaluate asset primaryKey to product by index at runtime.
/// </summary>
public class FormaProductIndexFinder : MonoBehaviour
{
    [Serializable]
    public class ProductIndexInfo
    {
        /// <summary>
        /// Addressable asset primaryKey (asset address) to find. Used in build at runtime.
        /// </summary>
        [SerializeField]
        public string keyToFind = string.Empty;

        /// <summary>
        /// Asset reference. Used in the Editor and in Editor runtime.
        /// </summary>
        [SerializeField]
        public AssetReference assetReference;

        /// <summary>
        /// Index from product settings list. Evaluated at runtime (Editor and in Build).
        /// </summary>
        [SerializeField]
        public int indexFound = -1;

        /// <summary>
        /// Your custom product identifier.
        /// </summary>
        [SerializeField]
        public string customProductId = string.Empty;
    }

    public bool debug;

    public List<ProductIndexInfo> productKeyToFind = new List<ProductIndexInfo>();

    void OnEnable()
    {
        GetComponent<ConfiguratorSettings>().settingsChanged += OnSettingsChanged;
    }

    void OnDisable()
    {
        GetComponent<ConfiguratorSettings>().settingsChanged -= OnSettingsChanged;
    }
    
    void Update()
    {
        if (!debug)
            return;

        if (Input.GetKeyDown(KeyCode.I))
        {
            Debug.Log($"Current loaded product custom id is {GetProductIndexInfoForCurrentProduct().customProductId}");
        }
    }
    
    void OnSettingsChanged()
    {
        var currentProducts = Configurator.s_ConfiguratorInstance.settings.productPrefabs;
        foreach (var productToFind in productKeyToFind)
        {
            for (var i = 0; i < currentProducts.Count; i++)
            {
                if (Addressable.GetCurrentLoadMode() == Addressable.LoadMode.EditorAssets)
                {
                    if (productToFind.assetReference != null)
                    {
#if UNITY_EDITOR
                        var aaSettings = AddressableAssetSettingsDefaultObject.Settings;
                        if (aaSettings)
                        {
                            var address = aaSettings.FindAssetEntry(productToFind.assetReference.AssetGUID).address;
                            if (address == currentProducts[i]?.key)
                                productToFind.indexFound = i;
                        }
#endif

                        // GUIDS check
                        if (productToFind.assetReference.RuntimeKey.ToString() == currentProducts[i]?.key)
                            productToFind.indexFound = i;
                    }
                }
                else if (Addressable.GetCurrentLoadMode() == Addressable.LoadMode.CatalogAssets)
                {
                    // Runtime Stuff
                    if (productToFind.keyToFind == currentProducts[i]?.key)
                        productToFind.indexFound = i;
                }
            }
        }

        if (debug)
            Debug.Log($"First Product index found (index: {productKeyToFind[0].indexFound}): {productKeyToFind[0].assetReference} with CustomProductId: {productKeyToFind[0].customProductId}");
    }

    public ProductIndexInfo GetProductIndexInfoForCurrentProduct()
    {
        var currentProductIndex = Configurator.s_ConfiguratorInstance.CurrentProductIndex;
        return productKeyToFind.FirstOrDefault(p => p.indexFound == currentProductIndex);
    }
}
